function NavRuleCondition()
{
	this.criteria = null;
	this.target = null;
	this.threshold = null; // range: NAV_THRESHOLD_NUMBER = [0,#children], NAV_THRESHOLD_PERCENT = [0,100]
	this.thresholdType = null;
	this.operator = null;
	this.useChildren = null;
	this.events = null;
	
	this.setDefaultValues();
}
NavRuleCondition.prototype.loadFromXMLNode = NavRuleConditionLoadFromXMLNode;
NavRuleCondition.prototype.setDefaultValues = NavRuleConditionSetDefaultValues;
NavRuleCondition.prototype.toString = NavRuleConditionToString;
NavRuleCondition.prototype.resolveReferences = NavRuleConditionResolveReferences;
NavRuleCondition.prototype.registerEvent = RegisterEvent;
NavRuleCondition.prototype.changed = NavRuleConditionChanged;
NavRuleCondition.prototype.evaluate = NavRuleConditionEvaluate;
NavRuleCondition.prototype.clone = NavRuleConditionClone;

function NavRuleConditionClone()
{
	var retVal = new NavRuleCondition();

	retVal.criteria = this.criteria;
	retVal.target = this.target.id;
	retVal.threshold = this.threshold;
	retVal.thresholdType = this.thresholdType;
	retVal.operator = this.operator;
	retVal.useChildren = this.useChildren;
	retVal.events =  new EventsObject(DEFAULT_RULE_EVENTS);
	
	return retVal;
}

function NavRuleConditionChanged(ref)
{
	this.events.raiseEvent("changed", this);
}

function NavRuleConditionResolveReferences(navRef)
{	
	this.target = navRef.getNodeById(this.target);
	if (this.useChildren)
	{
		for (var i=0;i<this.target.children.length;i++)
		{
			switch (this.criteria)
			{
				case NAV_CRITERIA_ATTEMPTED:this.target.children[i].registerEvent("attempted","changed",this,null,true);break;
				case NAV_CRITERIA_COMPLETED:this.target.children[i].registerEvent("completed","changed",this,null,true);break;
				case NAV_CRITERIA_SATISFIED:this.target.children[i].registerEvent("satisfied","changed",this,null,true);break;
			}
		}
	}
	else
	{
		switch (this.criteria)
		{
			case NAV_CRITERIA_ATTEMPTED:this.target.registerEvent("attempted","changed",this,null,true);break;
			case NAV_CRITERIA_COMPLETED:this.target.registerEvent("completed","changed",this,null,true);break;
			case NAV_CRITERIA_SATISFIED:this.target.registerEvent("satisfied","changed",this,null,true);break;
			case NAV_CRITERIA_SCORE:this.target.registerEvent("score","changed",this,null,true);break;
		}
	}
}

function NavRuleConditionToString()
{
	var str = "";
	var strCriteria = "";
	switch (this.criteria)
	{
		case NAV_CRITERIA_ATTEMPTED:strCriteria="attempted";break;
		case NAV_CRITERIA_COMPLETED:strCriteria="completed";break;
		case NAV_CRITERIA_SATISFIED:strCriteria="satisfied";break;
	}
	
	if (this.useChildren)
	{
		if (this.thresholdType == NAV_THRESHOLD_NUMBER)
		{
			str = "(" + ((this.operator==NAV_OPERATOR_NO_OP)?(""):("not ")) + "at least " + this.threshold + " children of " + this.target.id + " are " + strCriteria + ")";
		}
		else
		{
			str = "(" + ((this.operator==NAV_OPERATOR_NO_OP)?(""):("not ")) + "at least " + this.threshold + "% of the children of " + this.target.id + " are " + strCriteria + ")";
		}
	}
	else
	{
		if (this.criteria == NAV_CRITERIA_SCORE)
		{
			str = "(" + this.target.id + " score is " + ((this.operator==NAV_OPERATOR_NO_OP)?(""):("not ")) + ">= " + this.threshold + ")";
		}
		else
		{
			str = "(" + this.target.id + " is " + ((this.operator==NAV_OPERATOR_NO_OP)?(""):("not ")) + strCriteria + ")";
		}
	}
	
	return str;
}

function NavRuleConditionEvaluate()
{
	var accumulator = 0;
	var str = "";
	
	if (this.useChildren)
	{
		var childArray = this.target.children;
		if (this.target.selectedChildren.length > 0)
		{
			childArray = this.target.selectedChildren;
		}
		for (var i=0;i<childArray.length;i++)
		{
			switch (this.criteria)
			{
				case NAV_CRITERIA_ATTEMPTED:if (childArray[i].getAttempted()) accumulator++;break;
				case NAV_CRITERIA_COMPLETED:if (childArray[i].getCompleted()) accumulator++;break;
				case NAV_CRITERIA_SATISFIED:if (childArray[i].getSatisfied()) accumulator++;break;
			}
		}
		if (this.thresholdType == NAV_THRESHOLD_NUMBER)
		{
			str = "(" + ((this.operator==NAV_OPERATOR_NO_OP)?(""):("!")) + (accumulator >= this.threshold) + ")";
		}
		else
		{
			str = "(" + ((this.operator==NAV_OPERATOR_NO_OP)?(""):("!")) + ((accumulator/childArray.length*100) >= this.threshold) + ")";
		}
	}
	else
	{
		if (this.criteria == NAV_CRITERIA_SCORE)
		{
			str = "(" + ((this.operator==NAV_OPERATOR_NO_OP)?(""):("!")) + (this.target.getScore() >= this.threshold) + ")";
		}
		else
		{
			switch (this.criteria)
			{
				case NAV_CRITERIA_ATTEMPTED:str = "(" + ((this.operator==NAV_OPERATOR_NO_OP)?(""):("!")) + this.target.getAttempted() + ")";break;
				case NAV_CRITERIA_COMPLETED:str = "(" + ((this.operator==NAV_OPERATOR_NO_OP)?(""):("!")) + this.target.getCompleted() + ")";break;
				case NAV_CRITERIA_SATISFIED:str = "(" + ((this.operator==NAV_OPERATOR_NO_OP)?(""):("!")) + this.target.getSatisfied() + ")";break;
			}
		}
	}
	
	return eval(str);
}

function NavRuleConditionSetDefaultValues()
{
	this.criteria = NAV_CRITERIA_COMPLETED;
	this.target = null;
	this.threshold = 0; // range: NAV_THRESHOLD_NUMBER = [0,#children], NAV_THRESHOLD_PERCENT = [0,100]
	this.thresholdType = NAV_THRESHOLD_NUMBER;
	this.operator = NAV_OPERATOR_NO_OP;
	this.useChildren = false;
	this.events = new EventsObject(DEFAULT_RULE_EVENTS);
}

function NavRuleConditionLoadFromXMLNode(xmlNode)
{
	this.setDefaultValues();

	var criteria = xmlNode.tagName;
	switch (criteria.toLowerCase())
	{
		case "condattempted":this.criteria=NAV_CRITERIA_ATTEMPTED;this.threshold=0;break;
		case "condcompleted":this.criteria=NAV_CRITERIA_COMPLETED;this.threshold=0;break;
		case "condsatisfied":this.criteria=NAV_CRITERIA_SATISFIED;this.threshold=0;break;
		case "condminscore":
			this.criteria=NAV_CRITERIA_SCORE;
			this.threshold = floatParse(xmlNode.getAttribute("threshold"),SCORE_PRECISION);
			break;
		case "condpercentchildren":
			var criteria = xmlNode.getAttribute("criteria");
			switch (criteria.toLowerCase())
			{
				case "attempted":this.criteria=NAV_CRITERIA_ATTEMPTED;break;
				case "completed":this.criteria=NAV_CRITERIA_COMPLETED;break;
				case "satisfied":this.criteria=NAV_CRITERIA_SATISFIED;break;
				default:this.criteria=NAV_CRITERIA_COMPLETED;break;
			}
			this.thresholdType=NAV_THRESHOLD_PERCENT;
			this.threshold = floatParse(xmlNode.getAttribute("threshold"),SCORE_PRECISION);
			this.useChildren = true;
			break;
		case "condnumberchildren":
			var criteria = xmlNode.getAttribute("criteria");
			switch (criteria.toLowerCase())
			{
				case "attempted":this.criteria=NAV_CRITERIA_ATTEMPTED;break;
				case "completed":this.criteria=NAV_CRITERIA_COMPLETED;break;
				case "satisfied":this.criteria=NAV_CRITERIA_SATISFIED;break;
				default:this.criteria=NAV_CRITERIA_COMPLETED;break;
			}
			this.thresholdType=NAV_THRESHOLD_NUMBER;
			this.threshold = parseInt(xmlNode.getAttribute("threshold"));
			this.useChildren = true;
			break;
	}
	
	var operator = xmlNode.getAttribute("operator");
	switch ((operator+"").toLowerCase())
	{
		case "not":this.operator = NAV_OPERATOR_NOT;break;
		case "noop":this.operator = NAV_OPERATOR_NO_OP;break;
		default:this.operator = NAV_OPERATOR_NO_OP;break;
	}

	this.target = xmlNode.getAttribute("targetId");
}
